unit fROR_AboutDlg;
{
================================================================================
*
*       Package:        ROR - Clinical Case Registries
*       Date Created:   $Revision: 5 $  $Modtime: 4/12/07 3:48p $
*       Site:           Hines OIFO
*       Developers:                        @domain.ext
*                       Modified from Vitals Application by dan.petit@domain.ext
*	Description:    This is an About Dialog which displays data from the project
*                       VersionInfo block for the application.
*
*	Notes:          Delphi developer must add Keys VAReleaseDate and VANamespace
*                       Under Project|Options on the Version Info tab for the version,
*                       patch, and release date information to show up on the form.
*
================================================================================
*	$Archive: /CCR v1.5/Current/fROR_AboutDlg.pas $
*
* $History: fROR_AboutDlg.pas $
 * 
 * *****************  Version 5  *****************
 * User: Vhaishgavris Date: 4/12/07    Time: 3:49p
 * Updated in $/CCR v1.5/Current
 * 
 * *****************  Version 4  *****************
 * User: Vhaishgavris Date: 12/16/05   Time: 11:26a
 * Updated in $/CCR v1.0/Current
 * 
 * *****************  Version 3  *****************
 * User: Vhaishgavris Date: 10/14/04   Time: 3:49p
 * Updated in $/CCR v1.0/Current
 * 
 * *****************  Version 2  *****************
 * User: Vhaishgavris Date: 6/24/04    Time: 3:22p
 * Updated in $/CCR v1.0/Current
 * 
 * *****************  Version 1  *****************
 * User: Vhaishgavris Date: 9/11/03    Time: 5:03p
 * Created in $/ICR v3.0/Current
 *
 * *****************  Version 2  *****************
 * User: Vhaishandria Date: 8/01/03    Time: 9:48a
 * Updated in $/ICR v3.0/Current
 * Build 63
 * 
 * *****************  Version 1  *****************
 * User: Vhaishandria Date: 7/30/03    Time: 4:39p
 * Created in $/ICR v3.0/Current
 *
================================================================================
}

{$WARN UNIT_PLATFORM OFF}

interface

uses Windows,
  SysUtils,
  Classes,
  Graphics,
  Forms,
  Controls,
  StdCtrls,
  Buttons,
  ExtCtrls,
  Filectrl,
  uROR_Utilities, VA508AccessibilityManager;


const
  VersionInfoKeys: array[1..12] of string = (
    'CompanyName',
    'FileDescription',
    'FileVersion',
    'InternalName',
    'LegalCopyRight',
    'OriginalFileName',
    'ProductName',
    'ProductVersion',
    'Comments',
    'VAReleaseDate',
    'VANamespace',
    'VASourceInformation'
    );

  USEnglish = $040904E4;

type
  TVersionInfo = class(TComponent)
    (*
      Retrieves Version Info data about a given binary file.
    *)
  private
    FFileName: string;
    FLanguageID: DWord;
    FInfo: pointer;
    FInfoSize: longint;

    FCtlCompanyName: TControl;
    FCtlFileDescription: TControl;
    FCtlFileVersion: TControl;
    FCtlInternalName: TControl;
    FCtlLegalCopyRight: TControl;
    FCtlOriginalFileName: TControl;
    FCtlProductName: TControl;
    FCtlProductVersion: TControl;
    FCtlComments: TControl;
    FCtlVAReleaseDate: TControl;
    FCtlVANamespace: TControl;
    FCtlVASourceInformation: TControl;

    procedure SetFileName(Value: string);
    procedure SetVerProp(index: Integer; value: TControl);
    function GetVerProp(index: Integer): TControl;
    function GetIndexKey(index: Integer): string;
    function GetKey(const KName: string): string;
    function GetCompileDateTime: TDateTime;
    function GetVAPatchNumber: string;
    function GetVAServerVersion: string;
    function GetVATestVersion: Boolean;
    procedure Refresh;
  protected
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
  public
    property FileName: string read FFileName write SetFileName;
    property LanguageID: DWord read FLanguageID write FLanguageID;
    property CompileDateTime: TDateTime read GetCompileDateTime;

    property CompanyName: string index 1 read GetIndexKey;
    property FileDescription: string index 2 read GetIndexKey;
    property FileVersion: string index 3 read GetIndexKey;
    property InternalName: string index 4 read GetIndexKey;
    property LegalCopyRight: string index 5 read GetIndexKey;
    property OriginalFileName: string index 6 read GetIndexKey;
    property ProductName: string index 7 read GetIndexKey;
    property ProductVersion: string index 8 read GetIndexKey;
    property Comments: string index 9 read GetIndexKey;
    property VAReleaseDate: string index 10 read GetIndexKey;
    property VANamespace: string index 11 read GetIndexKey;
    property VASourceInformation: string index 12 read GetIndexKey;//AAN 08/05/2002
    property VAPatchNumber: string read GetVAPatchNumber;
    property VAServerVersion: string read GetVAServerVersion;
    property VATestVersion: Boolean read GetVATestVersion;

    constructor Create(AOwner: TComponent); override;
    (*
      Allocates memory and initializes variables for the object.
    *)

    destructor Destroy; override;
    (*
      Releases all memory allocated for the object.
    *)

    procedure OpenFile(FName: string);
    (*
      Uses method GetFileVersionInfo to retrieve version information for file
      FName.  If FName is blank, version information is obtained for the
      current executable (Application.ExeName).
    *)

    procedure Close;
    (*
      Releases memory allocated and clears all storage variables.
    *)

  published
    property CtlCompanyName: TControl index 1 read GetVerProp write SetVerProp;
    property CtlFileDescription: TControl index 2 read GetVerProp write SetVerProp;
    property CtlFileVersion: TControl index 3 read GetVerProp write SetVerProp;
    property CtlInternalName: TControl index 4 read GetVerProp write SetVerProp;
    property CtlLegalCopyRight: TControl index 5 read GetVerProp write SetVerProp;
    property CtlOriginalFileName: TControl index 6 read GetVerProp write SetVerProp;
    property CtlProductName: TControl index 7 read GetVerProp write SetVerProp;
    property CtlProductVersion: TControl index 8 read GetVerProp write SetVerProp;
    property CtlComments: TControl index 9 read GetVerProp write SetVerProp;
    property CtlVAReleaseDate: TControl index 10 read GetVerProp write SetVerProp;
    property CtlVANamespace: TControl index 11 read GetVerProp write SetVerProp;
    property CtlVASourceInfo: TControl index 12 read GetVerProp write SetVerProp;
  end;

  TFormAboutDlg = class(TForm)
    (*
      An About dialog which displays the Version Info data for the current
      application.
    *)
    Panel1: TPanel;
    pnlDividerBottom: TPanel;
    pnlComments: TPanel;
    Image: TImage;
    btnOK: TBitBtn;
    pnlLogo: TPanel;
    ApplicationImage: TImage;
    pnlDivderTop: TPanel;
    VA508AccessibilityManager1: TVA508AccessibilityManager;
    editWebLinkAppl1: TEdit;
    editWebLinkDoc1: TEdit;
    stxtClientVersionLabel: TStaticText;
    stxtClientVersion: TStaticText;
    stxtReleaseDateLabel: TStaticText;
    stxtReleaseDate: TStaticText;
    stxtPatchLabel: TStaticText;
    stxtPatch: TStaticText;
    stxtHelpFileLabel: TStaticText;
    stxtHelpFile: TStaticText;
    stxtCompileDateLabel: TStaticText;
    stxtCompileDate: TStaticText;
    stxtCRCLabel: TStaticText;
    stxtCRC: TStaticText;
    stxtSourceInfo: TStaticText;
    stxtHelpDesk: TStaticText;
    stxtCopyRight: TStaticText;
    stxtHelpDeskNumber: TStaticText;
    stxtWebLinkAppl1: TStaticText;
    stxtWebLinkDoc1: TStaticText;
    stxtSecurity: TStaticText;
    procedure FormCreate(Sender: TObject);
    procedure lblCommentsMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure OpenWebPage(PageToOpen: String);
    procedure KeyPressOnWebLink(Sender: TObject; var Key: Char);
    procedure MouseOnWebLink(Sender: TObject);
    procedure EnterWebLink(Sender: TObject);
    procedure ExitWebLink(Sender: TObject);
    procedure ClickOnWebLink(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure MouseOffWebLink(Sender: TObject);
    (*
      Uses TVersionInfo to read the Version Info for application.ExeName into
      the form's display fields.
    *)
  private
    { Private declarations }
    fComments: String; // ROR*1.5*19 2012-04-04 ==============================
  public
    { Public declarations }
    procedure AfterConstruction; override;
    procedure Execute;
    (*
      Creates the form and shows it modally.
    *)
  end;

var
  GMVAboutDlg: TFormAboutDlg;

implementation

{$R *.DFM}
uses
  Dialogs, Math, TypInfo, uROR_CRC, ShellApi, fROR_Main, uROR_Classes;

(*=== TVersionInfo Methods ==================================================*)

constructor TVersionInfo.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FLanguageID := USEnglish;
  SetFileName(EmptyStr);
end;

destructor TVersionInfo.Destroy;
begin
  if FInfoSize > 0 then
    FreeMem(FInfo, FInfoSize);
  inherited Destroy;
end;

procedure TVersionInfo.SetFileName(Value: string);
begin
  FFileName := Value;
  if Value = EmptyStr then
    FFileName := ExtractFileName(Application.ExeName);
  if csDesigning in ComponentState then
    Refresh
  else
    OpenFile(Value);
end;

procedure TVersionInfo.OpenFile(FName: string);
var
  vlen: DWord;
begin
  if FInfoSize > 0 then
    FreeMem(FInfo, FInfoSize);
  if Length(FName) <= 0 then
    FName := Application.ExeName;
  FInfoSize := GetFileVersionInfoSize(pchar(fname), vlen);
  if FInfoSize > 0 then
    begin
      GetMem(FInfo, FInfoSize);
      if not GetFileVersionInfo(pchar(fname), vlen, FInfoSize, FInfo) then
        raise Exception.Create('Cannot retrieve Version Information for ' + fname);
      Refresh;
    end;
end;

procedure TVersionInfo.Close;
begin
  if FInfoSize > 0 then
    FreeMem(FInfo, FInfoSize);
  FInfoSize := 0;
  FFileName := EmptyStr;
end;

const
  vqvFmt = '\StringFileInfo\%8.8x\%s';

function TVersionInfo.GetKey(const KName: string): string;
var
  vptr: pchar;
  vlen: DWord;
begin
  Result := EmptyStr;
  if FInfoSize <= 0 then
    exit;
  if VerQueryValue(FInfo, pchar(Format(vqvFmt, [FLanguageID, KName])), pointer(vptr), vlen) then
    Result := vptr;
end;

function TVersionInfo.GetIndexKey(index: Integer): string;
begin
  Result := GetKey(VersionInfoKeys[index]);
end;

function TVersionInfo.GetVerProp(index: Integer): TControl;
begin
  case index of
    1: Result := FCtlCompanyName;
    2: Result := FCtlFileDescription;
    3: Result := FCtlFileVersion;
    4: Result := FCtlInternalName;
    5: Result := FCtlLegalCopyRight;
    6: Result := FCtlOriginalFileName;
    7: Result := FCtlProductName;
    8: Result := FCtlProductVersion;
    9: Result := FCtlComments;
    10: Result := FCtlVAReleaseDate;
    11: Result := FCtlVANamespace;
    12: Result := FCtlVASourceInformation;
  else
    Result := nil;
  end;
end;

procedure TVersionInfo.SetVerProp(index: Integer; value: TControl);
begin
  case index of
    1: FCtlCompanyName := Value;
    2: FCtlFileDescription := Value;
    3: FCtlFileVersion := Value;
    4: FCtlInternalName := Value;
    5: FCtlLegalCopyRight := Value;
    6: FCtlOriginalFileName := Value;
    7: FCtlProductName := Value;
    8: FCtlProductVersion := Value;
    9: FCtlComments := Value;
    10: FCtlVAReleaseDate := Value;
    11: FCtlVANamespace := Value;
    12: FCtlVASourceInformation := Value;
  end;
  Refresh;
end;

procedure TVersionInfo.Notification(AComponent: TComponent; Operation: TOperation);
begin
  if Operation = opRemove then
    begin
      if AComponent = FCtlCompanyName then
        FCtlCompanyName := nil
      else if AComponent = FCtlFileDescription then
        FCtlFileDescription := nil
      else if AComponent = FCtlFileVersion then
        FCtlFileVersion := nil
      else if AComponent = FCtlInternalName then
        FCtlInternalName := nil
      else if AComponent = FCtlLegalCopyRight then
        FCtlLegalCopyRight := nil
      else if AComponent = FCtlOriginalFileName then
        FCtlOriginalFileName := nil
      else if AComponent = FCtlProductName then
        FCtlProductName := nil
      else if AComponent = FCtlProductVersion then
        FCtlProductVersion := nil
      else if AComponent = FCtlComments then
        FCtlComments := nil
      else if AComponent = FCtlVAReleaseDate then
        FCtlVAReleaseDate := nil
      else if AComponent = FCtlVANamespace then
        FCtlVANamespace := nil
      else if AComponent = FCtlVASourceInformation then
        FCtlVASourceInformation := nil;
    end;
end;

procedure TVersionInfo.Refresh;
var
  PropInfo: PPropInfo;

  procedure AssignText(Actl: TComponent; txt: string);
  begin
    if Assigned(ACtl) then
      begin
        PropInfo := GetPropInfo(ACtl.ClassInfo, 'Caption');
        if PropInfo <> nil then
          SetStrProp(ACtl, PropInfo, txt)
        else
          begin
            PropInfo := GetPropInfo(ACtl.ClassInfo, 'Text');
            if PropInfo <> nil then
              SetStrProp(ACtl, PropInfo, txt)
          end;
      end;
  end;

begin
  if csDesigning in ComponentState then
    begin
      AssignText(FCtlCompanyName, VersionInfoKeys[1]);
      AssignText(FCtlFileDescription, VersionInfoKeys[2]);
      AssignText(FCtlFileVersion, VersionInfoKeys[3]);
      AssignText(FCtlInternalName, VersionInfoKeys[4]);
      AssignText(FCtlLegalCopyRight, VersionInfoKeys[5]);
      AssignText(FCtlOriginalFileName, VersionInfoKeys[6]);
      AssignText(FCtlProductName, VersionInfoKeys[7]);
      AssignText(FCtlProductVersion, VersionInfoKeys[8]);
      AssignText(FCtlComments, VersionInfoKeys[9]);
      AssignText(FCtlVAReleaseDate, VersionInfoKeys[10]);
      AssignText(FCtlVANamespace, VersionInfoKeys[11]);
      AssignText(FCtlVASourceInformation, VersionInfoKeys[12]);
    end
  else
    begin
      AssignText(FCtlCompanyName, CompanyName);
      AssignText(FCtlFileDescription, FileDescription);
      AssignText(FCtlFileVersion, FileVersion);
      AssignText(FCtlInternalName, InternalName);
      AssignText(FCtlLegalCopyRight, LegalCopyRight);
      AssignText(FCtlOriginalFileName, OriginalFileName);
      AssignText(FCtlProductName, ProductName);
      AssignText(FCtlProductVersion, ProductVersion);
      AssignText(FCtlComments, Comments);
      AssignText(FCtlVAReleaseDate, VAReleaseDate);
      AssignText(FCtlVANamespace, VANamespace);
      AssignText(FCtlVASourceInformation, VASourceInformation);
    end;
end;

function TVersionInfo.GetCompileDateTime: TDateTime;
begin
  Result := FileDateToDateTime(FileAge(Application.ExeName));
end;

function TVersionInfo.GetVAPatchNumber: string;
begin
  if StrToIntDef(Piece(FileVersion, '.', 3), 0) > 0 then
    begin
      Result := VANamespace + '*' + VAServerVersion + '*' + Piece(FileVersion, '.', 3);
      {
      if StrToIntDef(Piece(FileVersion, '.', 4), 0) > 0 then
        Result := Result + 'T' + Piece(FileVersion, '.', 4);
      }
    end
  else
    Result := '';
end;

function TVersionInfo.GetVAServerVersion: string;
begin
  Result := Piece(FileVersion, '.', 1) +
    '.' + Piece(FileVersion, '.', 2);

  if StrToIntDef(Piece(FileVersion, '.', 3), 0) = 0 then
    if StrToIntDef(Piece(FileVersion, '.', 4), 0) > 0 then
      Result := Result + 'T' + Piece(FileVersion, '.', 4);
end;

function TVersionInfo.GetVATestVersion: Boolean;
begin
  Result := (StrToIntDef(Piece(FileVersion, '.', 4), 0) > 0);
end;

(*=== AboutDlg Methods ======================================================*)

procedure TFormAboutDlg.Execute;
begin
  with TFormAboutDlg.Create(self) do
  try
    pnlComments.Align := alClient;
    Image.Align := alClient;

    ShowModal;
  finally
    free;
  end;
end;

procedure TFormAboutDlg.FormCreate(Sender: TObject);
begin
  stxtHelpFile.Caption := MinimizeName(Application.HelpFile, self.Canvas, stxtHelpFile.Width);
  stxtHelpFile.Hint := 'Help file name: ' + Application.HelpFile;
  stxtHelpFile.ShowHint := (Pos('...', stxtHelpFile.Caption) > 0);

  stxtCRC.Caption := GetFileCRC32(Application.Exename);

  with TVersionInfo.Create(self) do
  try
    Self.Caption := 'About ' + ProductName;

    stxtSourceInfo.Caption := VASourceInformation;

    stxtClientVersion.Caption := FileVersion;
    if VAPatchNumber <> '' then
      begin
        stxtPatchLabel.Visible := True;
        stxtPatch.Caption := VAPatchNumber;
        stxtPatch.Visible := True;
      end;
    stxtCompileDate.Caption := FormatDateTime('mm/dd/yy hh:mm:ss',CompileDateTime);
    stxtReleaseDate.Caption := VAReleaseDate;
    stxtCopyRight.Caption := LegalCopyRight;
    fComments := Comments; // ROR*1.5*19 2012-04-04 ==========================
  finally
    free;
  end;
end;

procedure TFormAboutDlg.AfterConstruction;
var
  i: integer;
begin
  inherited;
  if CCRScreenReaderActive then
  begin
    for i := 0 to ComponentCount - 1 do
    begin
      if (Components[i] is TStaticText) then
      begin
        with TStaticText(Components[i]) do
          TabStop := true;
      end;
      if (Components[i] is TEdit) then
      begin
        with TEdit(Components[i]) do
          TabStop := true;
      end;
    end;
  end;
  SetFocusedControl(btnOK);
end;

procedure TFormAboutDlg.OpenWebPage(PageToOpen: String);
begin
  if Length(PageToOpen) > 0 then
  begin
    ShellExecute(Application.Handle, PChar('open'), PChar(PageToOpen),
      nil, nil, SW_SHOWNORMAL);
  end;
end;

procedure TFormAboutDlg.MouseOffWebLink(Sender: TObject);
begin
  Screen.Cursor := crDefault;
  (Sender as TEdit).Font.Style := [];
end;

procedure TFormAboutDlg.MouseOnWebLink(Sender: TObject);
begin
  Screen.Cursor := crHandPoint;
  (Sender as TEdit).Font.Style := [fsUnderline];
end;

procedure TFormAboutDlg.EnterWebLink(Sender: TObject);
begin
  (Sender as TEdit).Font.Style := [fsUnderline];
end;

procedure TFormAboutDlg.ExitWebLink(Sender: TObject);
begin
  (Sender as TEdit).Font.Style := [];
end;

procedure TFormAboutDlg.ClickOnWebLink(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  if (Button = mbLeft) and (Sender is TEdit) then
  begin
    OpenWebPage((Sender as TEdit).Text);
  end;
end;

procedure TFormAboutDlg.KeyPressOnWebLink(Sender: TObject; var Key: Char);
begin
  if Key = #13 then
  begin
    OpenWebPage((Sender as TEdit).Text);
    Key := #0;
  end;
  inherited;
end;

procedure TFormAboutDlg.lblCommentsMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var
  i: Integer;
const
  clText = clNavy;

procedure GrowRightCentered(aCanvas:TCanvas;aWait:Integer;
  aXMargin,Y,aWidth:Integer;aFColor:TColor;aText:String);
var
  ii,i: integer;
begin
  with aCanvas do
    begin
      Font.Style := [fsBold];
      ii := TextWidth(aText);
      ii := (aWidth - ii) div 2;
      aXMargin := ii;
      ii := TextHeight(aText);
      Font.Color := aFColor;
      for i := aXMargin to aWidth - aXMargin do
        begin
          FillRect(Rect(aXMargin,y,i,y+ii));
          TextRect(Rect(aXMargin,y,i,y+ii),aXMargin,y,aText);
          Application.ProcessMessages;
          sleep(aWait);
        end;
    end;
end;

procedure GrowRight(aCanvas:TCanvas;aWait:Integer;
  aXMargin,Y,aWidth:Integer;aFColor:TColor;aText:String);
var
  iLen,iWidth, iHeight,
  i: integer;
begin
  with aCanvas do
    begin
      Font.Style := [fsBold];
      iWidth := TextWidth(aText);
      iHeight := TextHeight(aText);
      Font.Color := aFColor;
      iLen := Min(iWidth, aWidth - aXMargin);
      for i := aXMargin to iLen do
        begin
          FillRect(Rect(aXMargin,y,i,y+iHeight));
          TextRect(Rect(aXMargin,y,i,y+iHeight),aXMargin,y,aText);
          Application.ProcessMessages;
          sleep(aWait);
        end;
    end;
end;

procedure SlideRight(aCanvas:TCanvas;aWait:Integer;
  aXMargin,Y,AwIDTH:Integer;aFColor:TColor;aText:String);
var
  iWidth, iHeight,
  i: integer;
begin
  with aCanvas do
    begin
      Font.Style := [fsBold];
      iWidth := TextWidth(aText);
      iHeight := TextHeight(aText);
      Font.Color := aFColor;
      for i := aXMargin to aXMargin+iWidth do
        begin
//          FillRect(Rect(aXMargin,y,i,y+ii));
          TextRect(Rect(aXMargin,y,i,y+iHeight),aXMargin,y,aText);
          Application.ProcessMessages;
          sleep(aWait);
        end;
    end;
end;

procedure SlideLeft(aCanvas:TCanvas;aWait:Integer;
  aXMargin,Y,aWidth:Integer;aFColor:TColor;aText:String);
var
  iT: Longint;
  iX,
  iLen,iWidth, iHeight,
  i: integer;
begin
  with aCanvas do
    begin
      Font.Style := [fsItalic];
      iWidth := TextWidth(aText);
      iHeight := TextHeight(aText);
      Font.Color := aFColor;
      iX := Max(aXMargin,aWidth-iWidth-aXMargin);
      i := 0;
      iLen := i;
      for i := aWidth - aXMargin downto iX do
        begin
          iLen := Min(iWidth,aWidth - aXMargin - i);
          FillRect(Rect(i,y,i+iLen+2,y+iHeight));
          TextRect(Rect(i,y,i+iLen,y+iHeight),i,y,aText);
          Application.ProcessMessages;
          if  (aWidth-aXMargin-i)> 0 then
            begin
              iT := aWait*(aWidth-aXMargin-i) div (aWidth-aXMargin);
              sleep(iT*iT);
            end;
        end;
      i := iX;
      FillRect(Rect(i,y,i+iLen+2,y+iHeight));
      Font.Style := [];
      TextRect(Rect(i,y,i+iLen,y+iHeight),i,y,aText);
      Application.ProcessMessages;
    end;
end;

begin
  if (Button = mbRight) and (ssShift in Shift) and (ssCtrl in Shift) then
    begin
      pnlComments.BringToFront;
      pnlComments.Visible := True;
      Image.Canvas.Brush.Color := clSilver;
      Image.Canvas.FillRect(Rect(0,0,Image.Width, Image.Height));
      i := 45;
      GrowRightCentered(Image.Canvas,0,30,i-25,Image.Width,clText,
        'Clinical Case Registries v1.5');
      GrowRightCentered(Image.Canvas,0,30,i-10,Image.Width,clText,
        'Development Team:');
      SlideRight(Image.Canvas, 10,30,i+20,Image.Width,clText,
        'Program Director:');
      SlideLeft(Image.Canvas, 30,30,i+20,Image.Width,clText,
        'Julius Chou');

      SlideRight(Image.Canvas, 10,30,i+35,Image.Width,clText,
        'Project Manager:');
      SlideLeft(Image.Canvas, 30,30,i+35,Image.Width,clText,
        'Jolie Renee');

      SlideRight(Image.Canvas,10,30,i+50,Image.Width,clText,
        'Windows GUI Development:');
      SlideLeft(Image.Canvas, 30,30,i+65,Image.Width,clText,
        'Andrey Andriyevskiy');
      SlideLeft(Image.Canvas, 30,30,i+80,Image.Width,clText,
        '               ');

      SlideRight(Image.Canvas,10,30,i+95,Image.Width,clText,
        'MUMPS Development:');
      SlideLeft(Image.Canvas, 30,30,i+110,Image.Width,clText,
        'Brend Hicks');
      SlideLeft(Image.Canvas, 30,30,i+125,Image.Width,clText,
        '               ');

      SlideRight(Image.Canvas,10,30,i+140,Image.Width,clText,
        'Technical Writer:');
      SlideLeft(Image.Canvas, 30,30,i+140,Image.Width,clText,
        '              ');

      SlideRight(Image.Canvas,10,30,i+155,Image.Width,clText,
        'SQA Analyst:');
      SlideLeft(Image.Canvas, 30,30,i+155,Image.Width,clText,
        'Pat Patrick');
      SlideRight(Image.Canvas, 10,30,i+170,Image.Width,clText,
        'Functional Analyst:');
      SlideLeft(Image.Canvas, 30,30,i+170,Image.Width,clText,
        'Jolie Renee');
      SlideRight(Image.Canvas, 10,30,i+185,Image.Width,clText,
        'Planner:');
      SlideLeft(Image.Canvas, 30,30,i+185,Image.Width,clText,
        'Joe Nicksich');

(*
// ROR*1.5*19 2012-04-04 ====================================================[
      GrowRightCentered(Image.Canvas,0,30,i+210,Image.Width,clText, 'Patch ROR*1.5*19 (ICD-10 Remediation)');
      SlideRight(Image.Canvas, 10,30,i+230,Image.Width,clText,'MUMPS Development:');
      SlideLeft(Image.Canvas, 30,30,i+230,Image.Width,clText, 'Kapil Gupta');
      SlideLeft(Image.Canvas, 30,30,i+245,Image.Width,clText, 'Jon W. Scott');
      SlideLeft(Image.Canvas, 30,30,i+260,Image.Width,clText, 'Jennifer Morrison');
      SlideRight(Image.Canvas,10,30,i+275,Image.Width,clText, 'Windows GUI Development:');
      SlideLeft(Image.Canvas, 30,30,i+275,Image.Width,clText, 'Andrey Andriyevskiy');
      SlideRight(Image.Canvas,10,30,i+290,Image.Width,clText, 'Technical Writer:');
      SlideLeft(Image.Canvas, 30,30,i+290,Image.Width,clText, 'Dawn Hoff');
      SlideLeft(Image.Canvas, 30,30,i+305,Image.Width,clText, 'Jennifer Nemetz');
// ROR*1.5*19 2012-04-04 ====================================================]
*)
      sleep(3000);
      pnlComments.Visible := False;
    end
  else
  if (Button = mbLeft) and (ssShift in Shift) and (ssCtrl in Shift) then
    ShowMessage(fComments);

end;

end.






